# This is a shell script that calls functions and scripts from
# tml@iki.fi's personal work envronment. It is not expected to be
# usable unmodified by others, and is included only for reference.

MOD=gtk+
VER=2.20.1
REV=3
ARCH=win32

THIS=${MOD}_${VER}-${REV}_${ARCH}

RUNZIP=${MOD}_${VER}-${REV}_${ARCH}.zip
DEVZIP=${MOD}-dev_${VER}-${REV}_${ARCH}.zip

HEX=`echo $THIS | md5sum | cut -d' ' -f1`
TARGET=c:/devel/target/$HEX

usedev
usemsvs6

(

set -x

patch --verbose --fuzz=0 -p1 <<'EOF' &&
commit a6c4833d9a307a1905516fb497dfaf0f141e1c0d
Author: Javier Jardón <jjardon@gnome.org>
Date:   Tue May 4 15:27:08 2010 +0200

    Remove the definition of libpixbufloader_gdip_png_la_*
    
    libstatic-pixbufloader-gdip-png.la should not be built or at least not
    included in STATIC_GDIPLUS_LIBS as we don't want to use the GDI+-based
    loader for PNG, because if we do, we can't get (or was it set?) the
    options of a PNG pixbuf that for instance some code in GIMP wants to do.
    
    Fixes https://bugzilla.gnome.org/show_bug.cgi?id=607839
---
 gdk-pixbuf/Makefile.am |   14 --------------
 1 files changed, 0 insertions(+), 14 deletions(-)

diff --git a/gdk-pixbuf/Makefile.am b/gdk-pixbuf/Makefile.am
index cf1769b..52fd710 100644
--- a/gdk-pixbuf/Makefile.am
+++ b/gdk-pixbuf/Makefile.am
@@ -244,9 +244,6 @@ libstatic_pixbufloader_gdip_gif_la_SOURCES = 	\
 libstatic_pixbufloader_gdip_jpeg_la_SOURCES = 	\
 	io-gdip-jpeg.c
 
-libstatic_pixbufloader_gdip_png_la_SOURCES = 	\
-	io-gdip-png.c
-
 libstatic_pixbufloader_gdip_tiff_la_SOURCES = 	\
 	io-gdip-tiff.c
 
@@ -327,17 +324,6 @@ libpixbufloader_gdip_jpeg_la_SOURCES = 	\
 	io-gdip-jpeg.c
 libpixbufloader_gdip_jpeg_la_LIBADD = $(module_libs) $(libole32)
 
-libpixbufloader_gdip_png_la_LDFLAGS = -avoid-version -module -no-undefined
-libpixbufloader_gdip_png_la_SOURCES = 	\
-	io-gdip-native.h		\
-	io-gdip-propertytags.h		\
-	io-gdip-utils.h			\
-	io-gdip-utils.c			\
-	io-gdip-animation.c		\
-	io-gdip-animation.h		\
-	io-gdip-png.c
-libpixbufloader_gdip_png_la_LIBADD = $(module_libs) $(libole32)
-
 libpixbufloader_gdip_tiff_la_LDFLAGS = -avoid-version -module -no-undefined
 libpixbufloader_gdip_tiff_la_SOURCES = 	\
 	io-gdip-native.h		\
commit fbd8c84acf71b13c109c8ffbcf3051c136d54fad
Author: Martin Schlemmer <11285613@nwu.ac.za>
Date:   Sat Aug 28 15:14:31 2010 +0200

    win32: cummulated backport of xp theme fixes from master
---
 gdk/gdk.symbols                        |    8 ++
 gdk/gdkinternals.h                     |    8 ++
 gdk/gdkwindow.c                        |  105 ++++++++++++++++++++++
 gdk/win32/gdkwin32.h                   |   11 +++
 gdk/win32/gdkwindow-win32.c            |   30 +++++++
 modules/engines/ms-windows/msw_style.c |  149 +++++++++++++-------------------
 modules/engines/ms-windows/xp_theme.c  |   79 +++++++++++------
 modules/engines/ms-windows/xp_theme.h  |   20 ++++
 8 files changed, 292 insertions(+), 118 deletions(-)

diff --git a/gdk/gdk.symbols b/gdk/gdk.symbols
index 548a063..d8901b3 100644
--- a/gdk/gdk.symbols
+++ b/gdk/gdk.symbols
@@ -1148,6 +1148,14 @@ gdk_window_destroy_notify
 
 #ifdef GDK_WINDOWING_WIN32
 #if IN_HEADER(__GDK_WIN32_H__)
+#if IN_FILE(__GDK_WINDOW_WIN32_C__)
+gdk_win32_window_is_win32
+gdk_win32_begin_direct_draw_libgtk_only
+gdk_win32_end_direct_draw_libgtk_only
+#endif
+#endif
+
+#if IN_HEADER(__GDK_WIN32_H__)
 #if IN_FILE(__GDK_WIN32ID_C__)
 gdk_win32_handle_table_lookup
 #endif
diff --git a/gdk/gdkinternals.h b/gdk/gdkinternals.h
index 822335f..d370693 100644
--- a/gdk/gdkinternals.h
+++ b/gdk/gdkinternals.h
@@ -423,6 +423,14 @@ void       _gdk_gc_set_clip_region_internal (GdkGC     *gc,
 					     gboolean reset_origin);
 GdkSubwindowMode _gdk_gc_get_subwindow (GdkGC *gc);
 
+GdkDrawable *_gdk_drawable_begin_direct_draw (GdkDrawable *drawable,
+					      GdkGC *gc,
+					      gpointer *priv_data,
+					      gint *x_offset_out,
+					      gint *y_offset_out);
+void         _gdk_drawable_end_direct_draw (gpointer priv_data);
+
+
 /*****************************************
  * Interfaces provided by windowing code *
  *****************************************/
diff --git a/gdk/gdkwindow.c b/gdk/gdkwindow.c
index 5c9219d..f29f57c 100644
--- a/gdk/gdkwindow.c
+++ b/gdk/gdkwindow.c
@@ -3636,6 +3636,111 @@ start_draw_helper (GdkDrawable *drawable,
      }                                                      \
   }
 
+#define BEGIN_DRAW_MACRO \
+  {
+
+#define END_DRAW_MACRO \
+  }
+
+typedef struct
+{
+  GdkDrawable *drawable;
+  GdkGC *gc;
+
+  gint x_offset;
+  gint y_offset;
+
+  gint clip_x;
+  gint clip_y;
+  gint ts_x;
+  gint ts_y;
+} DirectDrawInfo;
+
+GdkDrawable *
+_gdk_drawable_begin_direct_draw (GdkDrawable *drawable,
+				 GdkGC *gc,
+				 gpointer *priv_data,
+				 gint *x_offset_out,
+				 gint *y_offset_out)
+{
+  g_return_val_if_fail (priv_data != NULL, NULL);
+
+  GdkDrawable *out_impl = NULL;
+
+  *priv_data = NULL;
+
+  if (GDK_IS_PIXMAP (drawable))
+    {
+      /* We bypass the GdkPixmap functions, so do this ourself */
+      _gdk_gc_remove_drawable_clip (gc);
+
+      out_impl = drawable;
+
+      *x_offset_out = 0;
+      *y_offset_out = 0;
+    }
+  else
+    {
+      if (GDK_WINDOW_DESTROYED (drawable))
+        return NULL;
+
+      BEGIN_DRAW;
+
+      if (impl == NULL)
+        return NULL;
+
+      out_impl = impl;
+
+      *x_offset_out = x_offset;
+      *y_offset_out = y_offset;
+
+      DirectDrawInfo *priv = g_new (DirectDrawInfo, 1);
+
+      priv->drawable = impl;
+      priv->gc = gc;
+
+      priv->x_offset = x_offset;
+      priv->y_offset = y_offset;
+      priv->clip_x = old_clip_x;
+      priv->clip_y = old_clip_y;
+      priv->ts_x = old_ts_x;
+      priv->ts_y = old_ts_y;
+
+      *priv_data = (gpointer) priv;
+
+      END_DRAW_MACRO;
+    }
+
+  return out_impl;
+}
+
+void
+_gdk_drawable_end_direct_draw (gpointer priv_data)
+{
+  /* Its a GdkPixmap or the call to _gdk_drawable_begin_direct_draw failed. */
+  if (priv_data == NULL)
+    return;
+
+  DirectDrawInfo *priv = priv_data;
+  GdkGC *gc = priv->gc;
+
+  /* This is only for GdkWindows - if GdkPixmaps need any handling here in
+   * the future, then we should keep track of what type of drawable it is in
+   * DirectDrawInfo. */
+  BEGIN_DRAW_MACRO;
+
+  gint x_offset = priv->x_offset;
+  gint y_offset = priv->y_offset;
+  gint old_clip_x = priv->clip_x;
+  gint old_clip_y = priv->clip_y;
+  gint old_ts_x = priv->ts_x;
+  gint old_ts_y = priv->ts_y;
+
+  END_DRAW;
+
+  g_free (priv_data);
+}
+
 static GdkGC *
 gdk_window_create_gc (GdkDrawable     *drawable,
 		      GdkGCValues     *values,
diff --git a/gdk/win32/gdkwin32.h b/gdk/win32/gdkwin32.h
index bf1fd4b..7042d99 100644
--- a/gdk/win32/gdkwin32.h
+++ b/gdk/win32/gdkwin32.h
@@ -75,6 +75,9 @@ G_BEGIN_DECLS
 #endif
 
 
+/* Return true if the GdkWindow is a win32 implemented window */
+gboolean      gdk_win32_window_is_win32 (GdkWindow *window);
+
 /* Return the Gdk* for a particular HANDLE */
 gpointer      gdk_win32_handle_table_lookup (GdkNativeWindow handle);
 
@@ -107,6 +110,14 @@ GdkPixbuf    *gdk_win32_icon_to_pixbuf_libgtk_only (HICON hicon);
 HICON         gdk_win32_pixbuf_to_hicon_libgtk_only (GdkPixbuf *pixbuf);
 void          gdk_win32_set_modal_dialog_libgtk_only (HWND window);
 
+GdkDrawable  *gdk_win32_begin_direct_draw_libgtk_only (GdkDrawable *drawable,
+						       GdkGC *gc,
+						       gpointer *priv_data,
+						       gint *x_offset_out,
+						       gint *y_offset_out);
+void          gdk_win32_end_direct_draw_libgtk_only (gpointer priv_data);
+
+
 G_END_DECLS
 
 #endif /* __GDK_WIN32_H__ */
diff --git a/gdk/win32/gdkwindow-win32.c b/gdk/win32/gdkwindow-win32.c
index 598d291..9c7df31 100644
--- a/gdk/win32/gdkwindow-win32.c
+++ b/gdk/win32/gdkwindow-win32.c
@@ -3545,3 +3545,33 @@ gdk_window_impl_iface_init (GdkWindowImplIface *iface)
   iface->input_window_destroy = _gdk_input_window_destroy;
   iface->input_window_crossing = _gdk_input_crossing_event;
 }
+
+gboolean
+gdk_win32_window_is_win32 (GdkWindow *window)
+{
+  return GDK_WINDOW_IS_WIN32 (window);
+}
+
+GdkDrawable *
+gdk_win32_begin_direct_draw_libgtk_only (GdkDrawable *drawable,
+					 GdkGC *gc,
+					 gpointer *priv_data,
+					 gint *x_offset_out,
+					 gint *y_offset_out)
+{
+  GdkDrawable *impl;
+
+  impl = _gdk_drawable_begin_direct_draw (drawable,
+					  gc,
+					  priv_data,
+					  x_offset_out,
+					  y_offset_out);
+
+  return impl;
+}
+
+void
+gdk_win32_end_direct_draw_libgtk_only (gpointer priv_data)
+{
+  _gdk_drawable_end_direct_draw (priv_data);
+}
diff --git a/modules/engines/ms-windows/msw_style.c b/modules/engines/ms-windows/msw_style.c
index c69d1c7..189ec18 100755
--- a/modules/engines/ms-windows/msw_style.c
+++ b/modules/engines/ms-windows/msw_style.c
@@ -47,12 +47,6 @@
 #include "gdk/win32/gdkwin32.h"
 #endif
 
-static HDC get_window_dc (GtkStyle *style, GdkWindow *window,
-			  GtkStateType state_type, gint x, gint y, gint width,
-			  gint height, RECT *rect);
-static void release_window_dc (GtkStyle *style, GdkWindow *window,
-			       GtkStateType state_type);
-
 
 /* Default values, not normally used
  */
@@ -1113,8 +1107,9 @@ combo_box_draw_arrow (GtkStyle *style,
       DWORD border;
       RECT rect;
       HDC dc;
+      XpDCInfo dc_info;
 
-      dc = get_window_dc (style, window, state, area->x, area->y, area->width,
+      dc = get_window_dc (style, window, state, &dc_info, area->x, area->y, area->width,
 			  area->height, &rect);
       border = (GTK_TOGGLE_BUTTON (widget->parent)->
 		active ? DFCS_PUSHED | DFCS_FLAT : 0);
@@ -1122,7 +1117,7 @@ combo_box_draw_arrow (GtkStyle *style,
       InflateRect (&rect, 1, 1);
       DrawFrameControl (dc, &rect, DFC_SCROLL, DFCS_SCROLLDOWN | border);
 
-      release_window_dc (style, window, state);
+      release_window_dc (&dc_info);
 
       return TRUE;
     }
@@ -1270,8 +1265,9 @@ draw_expander (GtkStyle *style,
       RECT rect;
       HPEN pen;
       HGDIOBJ old_pen;
+      XpDCInfo dc_info;
 
-      dc = get_window_dc (style, window, state, x, y, expander_size,
+      dc = get_window_dc (style, window, state, &dc_info, x, y, expander_size,
 			  expander_size, &rect);
       FrameRect (dc, &rect, GetSysColorBrush (COLOR_GRAYTEXT));
       InflateRect (&rect, -1, -1);
@@ -1297,7 +1293,7 @@ draw_expander (GtkStyle *style,
 
       SelectObject (dc, old_pen);
       DeleteObject (pen);
-      release_window_dc (style, window, state);
+      release_window_dc (&dc_info);
     }
 
   if (area)
@@ -1510,6 +1506,7 @@ draw_arrow (GtkStyle *style,
   const gchar *name;
   HDC dc;
   RECT rect;
+  XpDCInfo dc_info;
 
   name = gtk_widget_get_name (widget);
 
@@ -1601,13 +1598,13 @@ draw_arrow (GtkStyle *style,
 	    {
 	      sanitize_size (window, &width, &height);
 
-	      dc = get_window_dc (style, window, state,
+	      dc = get_window_dc (style, window, state, &dc_info,
 				  box_x, box_y, box_width, box_height, &rect);
 	      DrawFrameControl (dc, &rect, DFC_SCROLL,
 				btn_type | (shadow ==
 					    GTK_SHADOW_IN ? (DFCS_PUSHED |
 							     DFCS_FLAT) : 0));
-	      release_window_dc (style, window, state);
+	      release_window_dc (&dc_info);
 	    }
 	}
     }
@@ -1716,50 +1713,6 @@ is_menu_tool_button_child (GtkWidget *wid)
   return FALSE;
 }
 
-HDC
-get_window_dc (GtkStyle *style, GdkWindow *window, GtkStateType state_type,
-	       gint x, gint y, gint width, gint height, RECT *rect)
-{
-  int xoff, yoff;
-  GdkDrawable *drawable;
-
-  if (!GDK_IS_WINDOW (window))
-    {
-      xoff = 0;
-      yoff = 0;
-      drawable = window;
-    }
-  else
-    {
-      gdk_window_get_internal_paint_info (window, &drawable, &xoff, &yoff);
-    }
-
-  rect->left = x - xoff;
-  rect->top = y - yoff;
-  rect->right = rect->left + width;
-  rect->bottom = rect->top + height;
-
-  return gdk_win32_hdc_get (drawable, style->dark_gc[state_type], 0);
-}
-
-void
-release_window_dc (GtkStyle *style, GdkWindow *window,
-		   GtkStateType state_type)
-{
-  GdkDrawable *drawable;
-
-  if (!GDK_IS_WINDOW (window))
-    {
-      drawable = window;
-    }
-  else
-    {
-      gdk_window_get_internal_paint_info (window, &drawable, NULL, NULL);
-    }
-
-  gdk_win32_hdc_release (drawable, style->dark_gc[state_type], 0);
-}
-
 static HPEN
 get_light_pen ()
 {
@@ -1823,6 +1776,7 @@ draw_menu_item (GdkWindow *window, GtkWidget *widget, GtkStyle *style,
   GtkMenuShell *bar;
   HDC dc;
   RECT rect;
+  XpDCInfo dc_info;
 
   if (xp_theme_is_active ())
     {
@@ -1835,14 +1789,14 @@ draw_menu_item (GdkWindow *window, GtkWidget *widget, GtkStyle *style,
     {
       bar = GTK_MENU_SHELL (parent);
 
-      dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+      dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
 
       if (state_type == GTK_STATE_PRELIGHT)
 	{
 	  draw_3d_border (dc, &rect, bar->active);
 	}
 
-      release_window_dc (style, window, state_type);
+      release_window_dc (&dc_info);
 
       return TRUE;
     }
@@ -1883,6 +1837,7 @@ draw_tool_button (GdkWindow *window, GtkWidget *widget, GtkStyle *style,
 {
   HDC dc;
   RECT rect;
+  XpDCInfo dc_info;
   gboolean is_toggled = FALSE;
 
   if (xp_theme_is_active ())
@@ -1905,7 +1860,7 @@ draw_tool_button (GdkWindow *window, GtkWidget *widget, GtkStyle *style,
       return FALSE;
     }
 
-  dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+  dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
   if (state_type == GTK_STATE_PRELIGHT)
     {
       if (is_toggled)
@@ -1927,7 +1882,7 @@ draw_tool_button (GdkWindow *window, GtkWidget *widget, GtkStyle *style,
       draw_3d_border (dc, &rect, TRUE);
     }
 
-  release_window_dc (style, window, state_type);
+  release_window_dc (&dc_info);
 
   return TRUE;
 }
@@ -1939,8 +1894,9 @@ draw_push_button (GdkWindow *window, GtkWidget *widget, GtkStyle *style,
 {
   HDC dc;
   RECT rect;
+  XpDCInfo dc_info;
 
-  dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+  dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
 
   if (GTK_IS_TOGGLE_BUTTON (widget))
     {
@@ -1980,7 +1936,7 @@ draw_push_button (GdkWindow *window, GtkWidget *widget, GtkStyle *style,
       DrawFrameControl (dc, &rect, DFC_BUTTON, DFCS_BUTTONPUSH);
     }
 
-  release_window_dc (style, window, state_type);
+  release_window_dc (&dc_info);
 }
 
 static void
@@ -1995,15 +1951,16 @@ draw_box (GtkStyle *style,
   if (is_combo_box_child (widget) && detail && !strcmp (detail, "button"))
     {
       RECT rect;
+      XpDCInfo dc_info;
       DWORD border;
       HDC dc;
       int cx;
 
       border = (GTK_TOGGLE_BUTTON (widget)->active ? DFCS_PUSHED | DFCS_FLAT : 0);
 
-      dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+      dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
       DrawFrameControl (dc, &rect, DFC_SCROLL, DFCS_SCROLLDOWN | border);
-      release_window_dc (style, window, state_type);
+      release_window_dc (&dc_info);
 
       if (xp_theme_is_active ()
 	  && xp_theme_draw (window, XP_THEME_ELEMENT_COMBOBUTTON, style, x, y,
@@ -2014,9 +1971,9 @@ draw_box (GtkStyle *style,
       width = cx;
 
 
-      dc = get_window_dc (style, window, state_type, x, y, width - cx, height, &rect);
+      dc = get_window_dc (style, window, state_type, &dc_info, x, y, width - cx, height, &rect);
       FillRect (dc, &rect, GetSysColorBrush (COLOR_WINDOW));
-      release_window_dc (style, window, state_type);
+      release_window_dc (&dc_info);
       return;
 	}
     }
@@ -2036,13 +1993,14 @@ draw_box (GtkStyle *style,
 	    {
 	      HDC dc;
 	      RECT rect;
-	      dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+	      XpDCInfo dc_info;
+	      dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
 
 	      DrawFrameControl (dc, &rect, DFC_BUTTON, DFCS_BUTTONPUSH |
 				(state_type ==
 				 GTK_STATE_ACTIVE ? (DFCS_PUSHED | DFCS_FLAT)
 				 : 0));
-	      release_window_dc (style, window, state_type);
+	      release_window_dc (&dc_info);
 	    }
 	}
       else if (is_toolbar_child (widget->parent)
@@ -2092,14 +2050,15 @@ draw_box (GtkStyle *style,
 			  style, x, y, width, height, state_type, area))
 	{
 	  RECT rect;
+	  XpDCInfo dc_info;
 	  HDC dc;
 
-	  dc = get_window_dc (style, window, state_type,
+	  dc = get_window_dc (style, window, state_type, &dc_info,
 			      x, y, width, height, &rect);
 	  DrawEdge (dc, &rect,
 		    state_type ==
 		    GTK_STATE_ACTIVE ? EDGE_SUNKEN : EDGE_RAISED, BF_RECT);
-	  release_window_dc (style, window, state_type);
+	  release_window_dc (&dc_info);
 	}
       return;
     }
@@ -2208,15 +2167,16 @@ draw_box (GtkStyle *style,
 	    {
 	      HDC dc;
 	      RECT rect;
+	      XpDCInfo dc_info;
 
 	      sanitize_size (window, &width, &height);
-	      dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+	      dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
 
 	      SetTextColor (dc, GetSysColor (COLOR_3DHILIGHT));
 	      SetBkColor (dc, GetSysColor (COLOR_BTNFACE));
 	      FillRect (dc, &rect, get_dither_brush ());
 
-	      release_window_dc (style, window, state_type);
+	      release_window_dc (&dc_info);
 
 	      return;
 	    }
@@ -2326,9 +2286,10 @@ draw_box (GtkStyle *style,
 	    {
 	      HBRUSH brush;
 	      RECT rect;
+	      XpDCInfo dc_info;
 	      HDC hdc;
 
-	      hdc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+	      hdc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
 
 	      brush = GetSysColorBrush (COLOR_3DDKSHADOW);
 
@@ -2340,7 +2301,7 @@ draw_box (GtkStyle *style,
 	      InflateRect (&rect, -1, -1);
 	      FillRect (hdc, &rect, (HBRUSH) (COLOR_INFOBK + 1));
 
-	      release_window_dc (style, window, state_type);
+	      release_window_dc (&dc_info);
 
 	      return;
 	    }
@@ -2771,10 +2732,11 @@ draw_tab_button (GtkStyle *style,
     {
       /* experimental tab-drawing code from mozilla */
       RECT rect;
+      XpDCInfo dc_info;
       HDC dc;
       gint32 aPosition;
 
-      dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+      dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
 
       if (gap_side == GTK_POS_TOP)
 	aPosition = BF_TOP;
@@ -2797,7 +2759,7 @@ draw_tab_button (GtkStyle *style,
       if (area)
 	gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
 
-      release_window_dc (style, window, state_type);
+      release_window_dc (&dc_info);
       return TRUE;
     }
 
@@ -2949,9 +2911,10 @@ draw_menu_border (GdkWindow *win, GtkStyle *style,
 		  gint x, gint y, gint width, gint height)
 {
   RECT rect;
+  XpDCInfo dc_info;
   HDC dc;
 
-  dc = get_window_dc (style, win, GTK_STATE_NORMAL, x, y, width, height, &rect);
+  dc = get_window_dc (style, win, GTK_STATE_NORMAL, &dc_info, x, y, width, height, &rect);
 
   if (!dc)
     return FALSE;
@@ -2965,7 +2928,7 @@ draw_menu_border (GdkWindow *win, GtkStyle *style,
       DrawEdge (dc, &rect, EDGE_RAISED, BF_RECT);
     }
 
-  release_window_dc (style, win, GTK_STATE_NORMAL);
+  release_window_dc (&dc_info);
 
   return TRUE;
 }
@@ -2987,10 +2950,11 @@ draw_shadow (GtkStyle *style,
 
       HDC dc;
       RECT rect;
+      XpDCInfo dc_info;
 
 
 
-      dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+      dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
       if (is_combo_box_child (widget))
         {
           FillRect (dc, &rect, GetSysColorBrush (COLOR_WINDOW));
@@ -3028,7 +2992,7 @@ draw_shadow (GtkStyle *style,
 	    }
 	}
 
-      release_window_dc (style, window, state_type);
+      release_window_dc (&dc_info);
 
       return;
     }
@@ -3042,12 +3006,13 @@ draw_shadow (GtkStyle *style,
 	{
 	  HDC dc;
 	  RECT rect;
+	  XpDCInfo dc_info;
 
-	  dc = get_window_dc (style, window, state_type,
+	  dc = get_window_dc (style, window, state_type, &dc_info,
 			      x, y, width, height, &rect);
 
 	  DrawEdge (dc, &rect, EDGE_SUNKEN, BF_RECT);
-	  release_window_dc (style, window, state_type);
+	  release_window_dc (&dc_info);
 	}
 
       return;
@@ -3090,6 +3055,7 @@ draw_shadow (GtkStyle *style,
 	{
 	  HDC dc;
 	  RECT rect;
+	  XpDCInfo dc_info;
 	  HGDIOBJ old_pen = NULL;
 	  GtkPositionType pos;
 
@@ -3142,7 +3108,7 @@ draw_shadow (GtkStyle *style,
 		}
 	    }
 
-	  dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+	  dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
 
 	  if (pos != GTK_POS_LEFT)
 	    {
@@ -3170,7 +3136,7 @@ draw_shadow (GtkStyle *style,
 	    }
 	  if (old_pen)
 	    SelectObject (dc, old_pen);
-	  release_window_dc (style, window, state_type);
+	  release_window_dc (&dc_info);
 	}
 
       return;
@@ -3322,13 +3288,14 @@ draw_resize_grip (GtkStyle *style,
       else
 	{
 	  RECT rect;
-	  HDC dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+	  XpDCInfo dc_info;
+	  HDC dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
 
 	  if (area)
 	    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], area);
 
 	  DrawFrameControl (dc, &rect, DFC_SCROLL, DFCS_SCROLLSIZEGRIP);
-	  release_window_dc (style, window, state_type);
+	  release_window_dc (&dc_info);
 
 	  if (area)
 	    gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
@@ -3354,6 +3321,7 @@ draw_handle (GtkStyle *style,
 {
   HDC dc;
   RECT rect;
+  XpDCInfo dc_info;
 
   if (is_toolbar_child (widget))
     {
@@ -3387,7 +3355,7 @@ draw_handle (GtkStyle *style,
 	  return;
 	}
 
-      dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+      dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
 
       if (orientation == GTK_ORIENTATION_VERTICAL)
 	{
@@ -3405,7 +3373,7 @@ draw_handle (GtkStyle *style,
 	}
 
       draw_3d_border (dc, &rect, FALSE);
-      release_window_dc (style, window, state_type);
+      release_window_dc (&dc_info);
       return;
     }
 
@@ -3478,6 +3446,7 @@ draw_focus (GtkStyle *style,
 {
   HDC dc;
   RECT rect;
+  XpDCInfo dc_info;
 
   if (!gtk_widget_get_can_focus (widget))
     {
@@ -3495,9 +3464,9 @@ draw_focus (GtkStyle *style,
       return;
     }
 
-  dc = get_window_dc (style, window, state_type, x, y, width, height, &rect);
+  dc = get_window_dc (style, window, state_type, &dc_info, x, y, width, height, &rect);
   DrawFocusRect (dc, &rect);
-  release_window_dc (style, window, state_type);
+  release_window_dc (&dc_info);
 /*
     parent_class->draw_focus (style, window, state_type,
 						     area, widget, detail, x, y, width, height);
diff --git a/modules/engines/ms-windows/xp_theme.c b/modules/engines/ms-windows/xp_theme.c
index 3d11935..506d9ed 100755
--- a/modules/engines/ms-windows/xp_theme.c
+++ b/modules/engines/ms-windows/xp_theme.c
@@ -849,6 +849,47 @@ xp_theme_map_gtk_state (XpThemeElement element, GtkStateType state)
   return ret;
 }
 
+HDC
+get_window_dc (GtkStyle *style,
+	       GdkWindow *window,
+	       GtkStateType state_type,
+	       XpDCInfo *dc_info_out,
+	       gint x, gint y, gint width, gint height,
+	       RECT *rect_out)
+{
+  GdkDrawable *drawable = NULL;
+  GdkGC *gc = style->dark_gc[state_type];
+  gint x_offset, y_offset;
+  
+  dc_info_out->data = NULL;
+  
+  drawable = gdk_win32_begin_direct_draw_libgtk_only (window,
+						      gc, &dc_info_out->data,
+						      &x_offset, &y_offset);
+  if (!drawable)
+    return NULL;
+
+  rect_out->left = x - x_offset;
+  rect_out->top = y - y_offset;
+  rect_out->right = rect_out->left + width;
+  rect_out->bottom = rect_out->top + height;
+  
+  dc_info_out->drawable = drawable;
+  dc_info_out->gc = gc;
+  dc_info_out->x_offset = x_offset;
+  dc_info_out->y_offset = y_offset;
+  
+  return gdk_win32_hdc_get (drawable, gc, 0);
+}
+
+void
+release_window_dc (XpDCInfo *dc_info)
+{
+  gdk_win32_hdc_release (dc_info->drawable, dc_info->gc, 0);
+
+  gdk_win32_end_direct_draw_libgtk_only (dc_info->data);
+}
+
 gboolean
 xp_theme_draw (GdkWindow *win, XpThemeElement element, GtkStyle *style,
 	       int x, int y, int width, int height,
@@ -856,9 +897,8 @@ xp_theme_draw (GdkWindow *win, XpThemeElement element, GtkStyle *style,
 {
   HTHEME theme;
   RECT rect, clip, *pClip;
-  int xoff, yoff;
   HDC dc;
-  GdkDrawable *drawable;
+  XpDCInfo dc_info;
   int part_state;
 
   if (!xp_theme_is_drawable (element))
@@ -869,28 +909,19 @@ xp_theme_draw (GdkWindow *win, XpThemeElement element, GtkStyle *style,
     return FALSE;
 
   /* FIXME: Recheck its function */
+  if (GDK_IS_WINDOW (win) && gdk_win32_window_is_win32 (win))
   enable_theme_dialog_texture_func (GDK_WINDOW_HWND (win), ETDT_ENABLETAB);
 
-  if (!GDK_IS_WINDOW (win))
-    {
-      xoff = 0;
-      yoff = 0;
-      drawable = win;
-    }
-  else
-    {
-      gdk_window_get_internal_paint_info (win, &drawable, &xoff, &yoff);
-    }
-
-  rect.left = x - xoff;
-  rect.top = y - yoff;
-  rect.right = rect.left + width;
-  rect.bottom = rect.top + height;
+  dc = get_window_dc (style, win, state_type, &dc_info,
+		      x, y, width, height,
+		      &rect);
+  if (!dc)
+    return FALSE;
 
   if (area)
     {
-      clip.left = area->x - xoff;
-      clip.top = area->y - yoff;
+      clip.left = area->x - dc_info.x_offset;
+      clip.top = area->y - dc_info.y_offset;
       clip.right = clip.left + area->width;
       clip.bottom = clip.top + area->height;
 
@@ -901,17 +932,12 @@ xp_theme_draw (GdkWindow *win, XpThemeElement element, GtkStyle *style,
       pClip = NULL;
     }
 
-  gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
-  dc = gdk_win32_hdc_get (drawable, style->dark_gc[state_type], 0);
-  if (!dc)
-    return FALSE;
-
   part_state = xp_theme_map_gtk_state (element, state_type);
 
   draw_theme_background_func (theme, dc, element_part_map[element],
 			      part_state, &rect, pClip);
 
-  gdk_win32_hdc_release (drawable, style->dark_gc[state_type], 0);
+  release_window_dc (&dc_info);
 
   return TRUE;
 }
@@ -919,9 +945,6 @@ xp_theme_draw (GdkWindow *win, XpThemeElement element, GtkStyle *style,
 gboolean
 xp_theme_is_active (void)
 {
-  /* Workaround for bug #598299 */
-  return FALSE;
-
   return use_xp_theme;
 }
 
diff --git a/modules/engines/ms-windows/xp_theme.h b/modules/engines/ms-windows/xp_theme.h
index 54d5de0..2d172f3 100755
--- a/modules/engines/ms-windows/xp_theme.h
+++ b/modules/engines/ms-windows/xp_theme.h
@@ -113,6 +113,26 @@ typedef enum
   XP_THEME_FONT_MESSAGE
 } XpThemeFont;
 
+typedef struct
+{
+  GdkDrawable *drawable;
+  GdkGC *gc;
+  
+  gint x_offset;
+  gint y_offset;
+  
+  /*< private >*/
+  gpointer data;
+} XpDCInfo;
+
+HDC get_window_dc (GtkStyle *style,
+		   GdkWindow *window,
+		   GtkStateType state_type,
+		   XpDCInfo *dc_info_out,
+		   gint x, gint y, gint width, gint height,
+		   RECT *rect_out);
+void release_window_dc (XpDCInfo *dc_info);
+
 void xp_theme_init (void);
 void xp_theme_reset (void);
 void xp_theme_exit (void);
commit ab5b246202beb5b28fdd84174f1f992aba5d76c0
Author: Tor Lillqvist <tml@iki.fi>
Date:   Thu Sep 2 14:30:59 2010 +0300

    Avoid potential DLL hijacking in ms-windows theme engine
    
    Load uxtheme.dll from an absolute path. A proper uxtheme.dll, if
    present, will always be in the Windows system directory, so load it
    from there.
---
 modules/engines/ms-windows/xp_theme.c |   28 +++++++++++++++++++++++++++-
 1 files changed, 27 insertions(+), 1 deletions(-)

diff --git a/modules/engines/ms-windows/xp_theme.c b/modules/engines/ms-windows/xp_theme.c
index 506d9ed..a0f333b 100755
--- a/modules/engines/ms-windows/xp_theme.c
+++ b/modules/engines/ms-windows/xp_theme.c
@@ -165,6 +165,8 @@ static const short element_part_map[XP_THEME_ELEMENT__SIZEOF] = {
   TKP_TICSVERT
 };
 
+#define UXTHEME_DLL "uxtheme.dll"
+
 static HINSTANCE uxtheme_dll = NULL;
 static HTHEME open_themes[XP_THEME_CLASS__SIZEOF];
 static gboolean use_xp_theme = FALSE;
@@ -219,12 +221,36 @@ xp_theme_close_open_handles (void)
 void
 xp_theme_init (void)
 {
+  char *buf;
+  char dummy;
+  int n, k;
+
   if (uxtheme_dll)
     return;
 
   memset (open_themes, 0, sizeof (open_themes));
 
-  uxtheme_dll = LoadLibrary ("uxtheme.dll");
+  n = GetSystemDirectory (&dummy, 0);
+
+  if (n <= 0)
+    return;
+
+  buf = g_malloc (n + 1 + strlen (UXTHEME_DLL));
+  k = GetSystemDirectory (buf, n);
+  
+  if (k == 0 || k > n)
+    {
+      g_free (buf);
+      return;
+    }
+
+  if (!G_IS_DIR_SEPARATOR (buf[strlen (buf) -1]))
+    strcat (buf, G_DIR_SEPARATOR_S);
+  strcat (buf, UXTHEME_DLL);
+
+  uxtheme_dll = LoadLibrary (buf);
+  g_free (buf);
+
   if (!uxtheme_dll)
     return;
 

commit 3d57c970470b182f644944f0f800c40c8b2f5adf
Author: Tor Lillqvist <tml@iki.fi>
Date:   Fri Sep 3 00:13:36 2010 +0300

    Just use TrackMouseEvent directly
    
    TrackMouseEvent is present in user32.dll in all Windows versions we
    support. No need to look it up dynamically. No need to fallback to
    _TrackMouseEvent from comctrl32.dll.
---
 gdk/win32/gdkevents-win32.c |   47 +++++++++++-------------------------------
 1 files changed, 13 insertions(+), 34 deletions(-)

diff --git a/gdk/win32/gdkevents-win32.c b/gdk/win32/gdkevents-win32.c
index 23dcd24..148c656 100644
--- a/gdk/win32/gdkevents-win32.c
+++ b/gdk/win32/gdkevents-win32.c
@@ -149,40 +149,19 @@ static void
 track_mouse_event (DWORD dwFlags,
 		   HWND  hwnd)
 {
-  typedef BOOL (WINAPI *PFN_TrackMouseEvent) (LPTRACKMOUSEEVENT);
-  static PFN_TrackMouseEvent p_TrackMouseEvent = NULL;
-  static gboolean once = FALSE;
-
-  if (!once)
-    {
-      HMODULE user32;
-      HINSTANCE commctrl32;
-
-      user32 = GetModuleHandle ("user32.dll");
-      if ((p_TrackMouseEvent = (PFN_TrackMouseEvent)GetProcAddress (user32, "TrackMouseEvent")) == NULL)
-        {
-          if ((commctrl32 = LoadLibrary ("commctrl32.dll")) != NULL)
-	    p_TrackMouseEvent = (PFN_TrackMouseEvent)
-	      GetProcAddress (commctrl32, "_TrackMouseEvent");
-        }
-      once = TRUE;
-    }
-
-  if (p_TrackMouseEvent)
-    {
-      TRACKMOUSEEVENT tme;
-      tme.cbSize = sizeof(TRACKMOUSEEVENT);
-      tme.dwFlags = dwFlags;
-      tme.hwndTrack = hwnd;
-      tme.dwHoverTime = HOVER_DEFAULT; /* not used */
-
-      if (!p_TrackMouseEvent (&tme))
-        WIN32_API_FAILED ("TrackMouseEvent");
-      else if (dwFlags == TME_LEAVE)
-        GDK_NOTE (EVENTS, g_print(" (TrackMouseEvent %p)", hwnd));
-      else if (dwFlags == TME_CANCEL)
-	GDK_NOTE (EVENTS, g_print(" (cancel TrackMouseEvent %p)", hwnd));
-    }
+  TRACKMOUSEEVENT tme;
+
+  tme.cbSize = sizeof(TRACKMOUSEEVENT);
+  tme.dwFlags = dwFlags;
+  tme.hwndTrack = hwnd;
+  tme.dwHoverTime = HOVER_DEFAULT; /* not used */
+
+  if (!TrackMouseEvent (&tme))
+    WIN32_API_FAILED ("TrackMouseEvent");
+  else if (dwFlags == TME_LEAVE)
+    GDK_NOTE (EVENTS, g_print(" (TrackMouseEvent %p)", hwnd));
+  else if (dwFlags == TME_CANCEL)
+    GDK_NOTE (EVENTS, g_print(" (cancel TrackMouseEvent %p)", hwnd));
 }
 
 gulong

commit c84425274a52fe153a8b34962ad84a8d116079e5
Author: Tor Lillqvist <tml@iki.fi>
Date:   Thu Sep 9 19:48:42 2010 +0300

    Avoid potential DLL hijacking in Wintab code
    
    Load Wintab32.dll from an absolute path. A proper Wintab32.dll, if
    present, should be in the Windows system directory, so load it from
    there.
---
 gdk/win32/gdkinput-win32.c |   26 ++++++++++++++++++++++++--
 1 files changed, 24 insertions(+), 2 deletions(-)

diff --git a/gdk/win32/gdkinput-win32.c b/gdk/win32/gdkinput-win32.c
index d02952d..5dec13c 100644
--- a/gdk/win32/gdkinput-win32.c
+++ b/gdk/win32/gdkinput-win32.c
@@ -37,6 +37,8 @@
 #include "gdkprivate-win32.h"
 #include "gdkinput-win32.h"
 
+#define WINTAB32_DLL "Wintab32.dll"
+
 #define PACKETDATA (PK_CONTEXT | PK_CURSOR | PK_BUTTONS | PK_X | PK_Y  | PK_NORMAL_PRESSURE | PK_ORIENTATION)
 /* We want everything in absolute mode */
 #define PACKETMODE (0)
@@ -334,12 +336,14 @@ _gdk_input_wintab_init_check (void)
   BOOL active;
   DWORD physid;
   AXIS axis_x, axis_y, axis_npressure, axis_or[3];
-  int i, k;
+  int i, k, n;
   int devix, cursorix;
   wchar_t devname[100], csrname[100];
   gchar *devname_utf8, *csrname_utf8;
   BOOL defcontext_done;
   HMODULE wintab32;
+  char *wintab32_dll_path;
+  char dummy;
 
   if (wintab_initialized)
     return;
@@ -351,7 +355,25 @@ _gdk_input_wintab_init_check (void)
   if (_gdk_input_ignore_wintab)
     return;
 
-  if ((wintab32 = LoadLibrary ("wintab32.dll")) == NULL)
+  n = GetSystemDirectory (&dummy, 0);
+
+  if (n <= 0)
+    return;
+
+  wintab32_dll_path = g_malloc (n + 1 + strlen (WINTAB32_DLL));
+  k = GetSystemDirectory (wintab32_dll_path, n);
+  
+  if (k == 0 || k > n)
+    {
+      g_free (wintab32_dll_path);
+      return;
+    }
+
+  if (!G_IS_DIR_SEPARATOR (wintab32_dll_path[strlen (wintab32_dll_path) -1]))
+    strcat (wintab32_dll_path, G_DIR_SEPARATOR_S);
+  strcat (wintab32_dll_path, WINTAB32_DLL);
+
+  if ((wintab32 = LoadLibrary (wintab32_dll_path)) == NULL)
     return;
 
   if ((p_WTInfoA = (t_WTInfoA) GetProcAddress (wintab32, "WTInfoA")) == NULL)

commit 76db5d6ce6cbe4ab7b4ec32e76baa4b91d25b5bf
Author: Tor Lillqvist <tml@iki.fi>
Date:   Thu Sep 9 20:35:30 2010 +0300

    Link directly to GDI+ API avoiding LoadLibrary
    
    Avoid potential DLL hijacking risks by not calling LoadLibrary() to
    load gdiplus.dll. As gdiplus.dll is a WinSxS (side-by-side) DLL we
    couldn't use the full path anyway as we don't know it.
    
    So just link to the GDI+ functions directly. Gdiplus.dll should be
    present on all Windows versions we support anyway. Some complexity
    added as MinGW doesn't come with an import library for gdiplus.dll, so
    we have to create one ourselves. But we call relatively few functions
    from gdiplus.dll so that is not a big deal.
    
    Note that despite being a side-by-side DLL (or "assembly"),
    gdiplus.dll is apparently a special case in that you don't need a
    manifest to tell the OS which version of gdiplus.dll you
    want. Weird.
---
 gdk-pixbuf/Makefile.am      |   31 +++++++++++-----
 gdk-pixbuf/gdiplus.def      |   26 +++++++++++++
 gdk-pixbuf/io-gdip-native.h |   63 ++++++++++++++------------------
 gdk-pixbuf/io-gdip-utils.c  |   84 ++-----------------------------------------
 4 files changed, 77 insertions(+), 127 deletions(-)

diff --git a/gdk-pixbuf/Makefile.am b/gdk-pixbuf/Makefile.am
index 52fd710..a8dbbff 100644
--- a/gdk-pixbuf/Makefile.am
+++ b/gdk-pixbuf/Makefile.am
@@ -206,6 +206,15 @@ libpixbufloader_qtif_la_LIBADD = $(module_libs)
 
 if BUILD_GDIPLUS_LOADERS
 
+# MinGW doesn't come with any import library for gdiplus.dll, so
+# create a partial one that's enough for our use.
+
+libgdiplus = libgdiplus.dll.a
+gdiplus_ldflag = -Wl,$(libgdiplus)
+
+libgdiplus.dll.a: gdiplus.def
+	$(DLLTOOL) --kill-at --dllname gdiplus.dll --input-def gdiplus.def --output-lib $@
+
 if INCLUDE_GDIPLUS
 
 # When building the GDI+ loader statically, we put the "common" objects
@@ -258,7 +267,7 @@ GDIPLUS_LIBS = \
 	libpixbufloader-gdip-jpeg.la \
 	libpixbufloader-gdip-tiff.la
 
-libpixbufloader_gdip_ico_la_LDFLAGS = -avoid-version -module -no-undefined
+libpixbufloader_gdip_ico_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 libpixbufloader_gdip_ico_la_SOURCES = 	\
 	io-gdip-native.h		\
 	io-gdip-propertytags.h		\
@@ -269,7 +278,7 @@ libpixbufloader_gdip_ico_la_SOURCES = 	\
 	io-gdip-ico.c
 libpixbufloader_gdip_ico_la_LIBADD = $(module_libs) $(libole32)
 
-libpixbufloader_gdip_wmf_la_LDFLAGS = -avoid-version -module -no-undefined
+libpixbufloader_gdip_wmf_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 libpixbufloader_gdip_wmf_la_SOURCES = 	\
 	io-gdip-native.h		\
 	io-gdip-propertytags.h		\
@@ -280,7 +289,7 @@ libpixbufloader_gdip_wmf_la_SOURCES = 	\
 	io-gdip-wmf.c
 libpixbufloader_gdip_wmf_la_LIBADD = $(module_libs) $(libole32)
 
-libpixbufloader_gdip_emf_la_LDFLAGS = -avoid-version -module -no-undefined
+libpixbufloader_gdip_emf_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 libpixbufloader_gdip_emf_la_SOURCES = 	\
 	io-gdip-native.h		\
 	io-gdip-propertytags.h		\
@@ -291,7 +300,7 @@ libpixbufloader_gdip_emf_la_SOURCES = 	\
 	io-gdip-emf.c
 libpixbufloader_gdip_emf_la_LIBADD = $(module_libs) $(libole32)
 
-libpixbufloader_gdip_bmp_la_LDFLAGS = -avoid-version -module -no-undefined
+libpixbufloader_gdip_bmp_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 libpixbufloader_gdip_bmp_la_SOURCES = 	\
 	io-gdip-native.h		\
 	io-gdip-propertytags.h		\
@@ -302,7 +311,7 @@ libpixbufloader_gdip_bmp_la_SOURCES = 	\
 	io-gdip-bmp.c
 libpixbufloader_gdip_bmp_la_LIBADD = $(module_libs) $(libole32)
 
-libpixbufloader_gdip_gif_la_LDFLAGS = -avoid-version -module -no-undefined
+libpixbufloader_gdip_gif_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 libpixbufloader_gdip_gif_la_SOURCES = 	\
 	io-gdip-native.h		\
 	io-gdip-propertytags.h		\
@@ -313,7 +322,7 @@ libpixbufloader_gdip_gif_la_SOURCES = 	\
 	io-gdip-gif.c
 libpixbufloader_gdip_gif_la_LIBADD = $(module_libs) $(libole32)
 
-libpixbufloader_gdip_jpeg_la_LDFLAGS = -avoid-version -module -no-undefined
+libpixbufloader_gdip_jpeg_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 libpixbufloader_gdip_jpeg_la_SOURCES = 	\
 	io-gdip-native.h		\
 	io-gdip-propertytags.h		\
@@ -324,7 +333,7 @@ libpixbufloader_gdip_jpeg_la_SOURCES = 	\
 	io-gdip-jpeg.c
 libpixbufloader_gdip_jpeg_la_LIBADD = $(module_libs) $(libole32)
 
-libpixbufloader_gdip_tiff_la_LDFLAGS = -avoid-version -module -no-undefined
+libpixbufloader_gdip_tiff_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 libpixbufloader_gdip_tiff_la_SOURCES = 	\
 	io-gdip-native.h		\
 	io-gdip-propertytags.h		\
@@ -557,11 +566,12 @@ libgdk_pixbuf_2_0_la_LDFLAGS = \
 	-version-info $(LT_VERSION_INFO)	\
 	$(LIBTOOL_EXPORT_OPTIONS)		\
 	$(no_undefined)				\
-	$(gdk_pixbuf_symbols)
+	$(gdk_pixbuf_symbols)			\
+	$(gdiplus_ldflag)
 
 
 libgdk_pixbuf_2_0_la_LIBADD = pixops/libpixops.la $(builtin_objs) $(GDK_PIXBUF_DEP_LIBS) $(libole32)
-libgdk_pixbuf_2_0_la_DEPENDENCIES = pixops/libpixops.la $(builtin_objs) $(gdk_pixbuf_def) $(gdk_pixbuf_win32_res)
+libgdk_pixbuf_2_0_la_DEPENDENCIES = pixops/libpixops.la $(builtin_objs) $(gdk_pixbuf_def) $(gdk_pixbuf_win32_res) $(libgdiplus)
 
 gdk_pixbuf_headers = 			\
 	gdk-pixbuf.h			\
@@ -673,7 +683,8 @@ EXTRA_DIST +=					\
 	gdk-pixbuf-marshal.list			\
 	gdk-pixbuf-enum-types.c.template	\
 	gdk-pixbuf-enum-types.h.template	\
-	gen-color-table.pl
+	gen-color-table.pl			\
+	gdiplus.def
 
 if HAVE_INTROSPECTION
 
diff --git a/gdk-pixbuf/gdiplus.def b/gdk-pixbuf/gdiplus.def
new file mode 100644
index 0000000..a32a4a1
--- /dev/null
+++ b/gdk-pixbuf/gdiplus.def
@@ -0,0 +1,26 @@
+EXPORTS
+	GdiplusStartup@12
+	GdipCreateBitmapFromStream@8
+	GdipBitmapGetPixel@16
+	GdipGetImageWidth@8
+	GdipGetImageHeight@8
+	GdipDisposeImage@4
+	GdipGetImageFlags@8
+	GdipImageGetFrameCount@12
+	GdipImageSelectActiveFrame@12
+	GdipGetPropertyItemSize@12
+	GdipGetPropertyItem@16
+	GdipCreateBitmapFromScan0@24
+	GdipSaveImageToStream@16
+	GdipGetImageEncoders@12
+	GdipGetImageEncodersSize@8
+	GdipBitmapSetPixel@16
+	GdipDrawImageI@16
+	GdipGetImageGraphicsContext@8
+	GdipFlush@8
+	GdipGraphicsClear@8
+	GdipBitmapSetResolution@12
+	GdipGetImageHorizontalResolution@8
+	GdipGetImageVerticalResolution@8
+	GdipLoadImageFromStream@8
+	GdipDeleteGraphics@4
diff --git a/gdk-pixbuf/io-gdip-native.h b/gdk-pixbuf/io-gdip-native.h
index 24ea824..417a747 100644
--- a/gdk-pixbuf/io-gdip-native.h
+++ b/gdk-pixbuf/io-gdip-native.h
@@ -224,41 +224,32 @@ typedef struct _GpRect GpRect;
 #define IStream_SetSize(This,size) (This)->lpVtbl->SetSize(This,size)
 #endif
 
-typedef GpStatus (WINGDIPAPI* GdiplusStartupFunc) (gpointer, const gpointer, gpointer);
-typedef GpStatus (WINGDIPAPI* GdipCreateBitmapFromStreamFunc) (gpointer, GpBitmap**);
-typedef GpStatus (WINGDIPAPI* GdipBitmapGetPixelFunc) (GpBitmap*, gint x, gint y, ARGB*);
-typedef GpStatus (WINGDIPAPI* GdipGetImageWidthFunc) (GpImage*, guint*);
-typedef GpStatus (WINGDIPAPI* GdipGetImageHeightFunc) (GpImage*, guint*);
-typedef GpStatus (WINGDIPAPI* GdipDisposeImageFunc) (GpImage*);
-typedef GpStatus (WINGDIPAPI* GdipGetImageFlagsFunc) (GpImage *, guint*);
-typedef GpStatus (WINGDIPAPI* GdipImageGetFrameCountFunc) (GpImage *image, const GUID* dimensionID, UINT* count);
-typedef GpStatus (WINGDIPAPI* GdipImageSelectActiveFrameFunc) (GpImage *image, const GUID* dimensionID, UINT frameIndex);
-typedef GpStatus (WINGDIPAPI* GdipGetPropertyItemSizeFunc) (GpImage *image, int propId, guint* size);
-typedef GpStatus (WINGDIPAPI* GdipGetPropertyItemFunc) (GpImage *image, int propId, guint propSize, PropertyItem* buffer);
-typedef GpStatus (WINGDIPAPI* GdipGetPropertyCountFunc) (GpImage *image, guint* numOfProperty);
-typedef GpStatus (WINGDIPAPI* GdipGetPropertyIdListFunc) (GpImage *image, guint numOfProperty, PROPID* list);
-typedef GpStatus (WINGDIPAPI* GdipCreateBitmapFromScan0Func) (INT width, INT height, INT stride, PixelFormat format, BYTE* scan0, 
-                                                              GpBitmap** bitmap);
-typedef GpStatus (WINGDIPAPI* GdipSaveImageToStreamFunc) (GpImage *image, IStream* stream, const CLSID* clsidEncoder, 
-                                                          const EncoderParameters* encoderParams);
-
-typedef GpStatus (WINGDIPAPI* GdipGetImageEncodersFunc) (UINT numEncoders, UINT size, ImageCodecInfo *encoders);
-typedef GpStatus (WINGDIPAPI* GdipGetImageEncodersSizeFunc) (UINT *numEncoders, UINT *size);
-typedef GpStatus (WINGDIPAPI* GdipBitmapSetPixelFunc) (GpBitmap* bitmap, INT x, INT y, ARGB color);
-
-typedef GpStatus (WINGDIPAPI* GdipDrawImageIFunc) (GpGraphics *graphics, GpImage *image, INT x, INT y);
-typedef GpStatus (WINGDIPAPI* GdipGetImageGraphicsContextFunc) (GpImage *image, GpGraphics **graphics);
-typedef GpStatus (WINGDIPAPI* GdipFlushFunc) (GpGraphics *graphics, INT intention);
-typedef GpStatus (WINGDIPAPI* GdipGraphicsClearFunc) (GpGraphics *graphics, ARGB color);
-typedef GpStatus (WINGDIPAPI* GdipBitmapSetResolutionFunc) (GpBitmap* bitmap, float xdpi, float ydpi);
-typedef GpStatus (WINGDIPAPI* GdipGetImageHorizontalResolutionFunc) (GpImage *image, float *resolution);
-typedef GpStatus (WINGDIPAPI* GdipGetImageVerticalResolutionFunc) (GpImage *image, float *resolution);
-typedef GpStatus (WINGDIPAPI* GdipLoadImageFromStreamFunc) (IStream* stream, GpImage **image);
-typedef GpStatus (WINGDIPAPI* GdipDeleteGraphicsFunc) (GpGraphics *graphics);
-
-typedef GpStatus (WINGDIPAPI* GdipBitmapLockBitsFunc) (GpBitmap* bitmap, const GpRect* rect, UINT flags, PixelFormat format, BitmapData* lockedBitmapData);
-typedef GpStatus (WINGDIPAPI* GdipBitmapUnlockBitsFunc) (GpBitmap* bitmap, BitmapData* lockedBitmapData);
-typedef GpStatus (WINGDIPAPI* GdipGetImagePixelFormatFunc) (GpImage *image, PixelFormat *format);
-typedef GpStatus (WINGDIPAPI* GdipCloneBitmapAreaIFunc) (INT x, INT y, INT width, INT height, PixelFormat format, GpBitmap *srcBitmap, GpBitmap **dstBitmap);
+GpStatus WINGDIPAPI GdiplusStartup (gpointer, const gpointer, gpointer);
+GpStatus WINGDIPAPI GdipCreateBitmapFromStream (gpointer, GpBitmap**);
+GpStatus WINGDIPAPI GdipBitmapGetPixel (GpBitmap*, gint x, gint y, ARGB*);
+GpStatus WINGDIPAPI GdipGetImageWidth (GpImage*, guint*);
+GpStatus WINGDIPAPI GdipGetImageHeight (GpImage*, guint*);
+GpStatus WINGDIPAPI GdipDisposeImage (GpImage*);
+GpStatus WINGDIPAPI GdipGetImageFlags (GpImage *, guint*);
+GpStatus WINGDIPAPI GdipImageGetFrameCount (GpImage *image, const GUID* dimensionID, UINT* count);
+GpStatus WINGDIPAPI GdipImageSelectActiveFrame (GpImage *image, const GUID* dimensionID, UINT frameIndex);
+GpStatus WINGDIPAPI GdipGetPropertyItemSize (GpImage *image, int propId, guint* size);
+GpStatus WINGDIPAPI GdipGetPropertyItem (GpImage *image, int propId, guint propSize, PropertyItem* buffer);
+GpStatus WINGDIPAPI GdipCreateBitmapFromScan0 (INT width, INT height, INT stride, PixelFormat format, BYTE* scan0, 
+                                               GpBitmap** bitmap);
+GpStatus WINGDIPAPI GdipSaveImageToStream (GpImage *image, IStream* stream, const CLSID* clsidEncoder, 
+                                           const EncoderParameters* encoderParams);
+GpStatus WINGDIPAPI GdipGetImageEncoders (UINT numEncoders, UINT size, ImageCodecInfo *encoders);
+GpStatus WINGDIPAPI GdipGetImageEncodersSize (UINT *numEncoders, UINT *size);
+GpStatus WINGDIPAPI GdipBitmapSetPixel (GpBitmap* bitmap, INT x, INT y, ARGB color);
+GpStatus WINGDIPAPI GdipDrawImageI (GpGraphics *graphics, GpImage *image, INT x, INT y);
+GpStatus WINGDIPAPI GdipGetImageGraphicsContext (GpImage *image, GpGraphics **graphics);
+GpStatus WINGDIPAPI GdipFlush (GpGraphics *graphics, INT intention);
+GpStatus WINGDIPAPI GdipGraphicsClear (GpGraphics *graphics, ARGB color);
+GpStatus WINGDIPAPI GdipBitmapSetResolution (GpBitmap* bitmap, float xdpi, float ydpi);
+GpStatus WINGDIPAPI GdipGetImageHorizontalResolution (GpImage *image, float *resolution);
+GpStatus WINGDIPAPI GdipGetImageVerticalResolution (GpImage *image, float *resolution);
+GpStatus WINGDIPAPI GdipLoadImageFromStream (IStream* stream, GpImage **image);
+GpStatus WINGDIPAPI GdipDeleteGraphics (GpGraphics *graphics);
 
 #endif
diff --git a/gdk-pixbuf/io-gdip-utils.c b/gdk-pixbuf/io-gdip-utils.c
index b8fa72b..8f62d75 100644
--- a/gdk-pixbuf/io-gdip-utils.c
+++ b/gdk-pixbuf/io-gdip-utils.c
@@ -43,38 +43,6 @@ struct _GdipContext {
 };
 typedef struct _GdipContext GdipContext;
 
-static GdiplusStartupFunc GdiplusStartup;
-static GdipCreateBitmapFromStreamFunc GdipCreateBitmapFromStream;
-static GdipBitmapGetPixelFunc GdipBitmapGetPixel;
-static GdipGetImageHeightFunc GdipGetImageHeight;
-static GdipDisposeImageFunc GdipDisposeImage;
-static GdipGetImageFlagsFunc GdipGetImageFlags;
-static GdipGetImageWidthFunc GdipGetImageWidth;
-static GdipImageGetFrameCountFunc GdipImageGetFrameCount;
-static GdipImageSelectActiveFrameFunc GdipImageSelectActiveFrame;
-static GdipGetPropertyItemSizeFunc GdipGetPropertyItemSize;
-static GdipGetPropertyItemFunc GdipGetPropertyItem;
-static GdipGetPropertyCountFunc GdipGetPropertyCount;
-static GdipGetPropertyIdListFunc GdipGetPropertyIdList;
-static GdipCreateBitmapFromScan0Func GdipCreateBitmapFromScan0;
-static GdipSaveImageToStreamFunc GdipSaveImageToStream;
-static GdipBitmapSetPixelFunc GdipBitmapSetPixel;
-static GdipDrawImageIFunc GdipDrawImageI;
-static GdipGetImageGraphicsContextFunc GdipGetImageGraphicsContext;
-static GdipFlushFunc GdipFlush;
-static GdipGraphicsClearFunc GdipGraphicsClear;
-static GdipBitmapSetResolutionFunc GdipBitmapSetResolution;
-static GdipGetImageHorizontalResolutionFunc GdipGetImageHorizontalResolution;
-static GdipGetImageVerticalResolutionFunc GdipGetImageVerticalResolution;
-static GdipLoadImageFromStreamFunc GdipLoadImageFromStream;
-static GdipDeleteGraphicsFunc GdipDeleteGraphics;
-static GdipGetImageEncodersFunc GdipGetImageEncoders;
-static GdipGetImageEncodersSizeFunc GdipGetImageEncodersSize;
-static GdipBitmapLockBitsFunc GdipBitmapLockBits;
-static GdipBitmapUnlockBitsFunc GdipBitmapUnlockBits;
-static GdipGetImagePixelFormatFunc GdipGetImagePixelFormat;
-static GdipCloneBitmapAreaIFunc GdipCloneBitmapAreaI;
-
 DEFINE_GUID(FrameDimensionTime, 0x6aedbd6d,0x3fb5,0x418a,0x83,0xa6,0x7f,0x45,0x22,0x9d,0xc8,0x72);
 DEFINE_GUID(FrameDimensionPage, 0x7462dc86,0x6180,0x4c7e,0x8e,0x3f,0xee,0x73,0x33,0xa7,0xa4,0x83);
 
@@ -131,58 +99,12 @@ gdip_init (void)
 {
   GdiplusStartupInput input;
   ULONG_PTR gdiplusToken = 0;
-  static HINSTANCE gdipluslib = NULL;
+  static gboolean beenhere = FALSE;
 
-  if (!gdipluslib)
-    gdipluslib = LoadLibrary ("gdiplus.dll");
-  else
+  if (beenhere)
     return TRUE; /* gdip_init() is idempotent */
 
-  if (!gdipluslib)
-    return FALSE;
-
-#define LOOKUP(func) \
-  G_STMT_START { \
-    func = (func##Func) GetProcAddress (gdipluslib, #func); \
-    if (!func) {\
-      g_warning ("Couldn't find GDI+ function %s\n", #func); \
-      return FALSE; \
-    } \
-  } G_STMT_END
-
-  LOOKUP (GdiplusStartup);
-  LOOKUP (GdipCreateBitmapFromStream);
-  LOOKUP (GdipBitmapGetPixel);
-  LOOKUP (GdipGetImageHeight);
-  LOOKUP (GdipDisposeImage);
-  LOOKUP (GdipGetImageFlags);
-  LOOKUP (GdipGetImageWidth);
-  LOOKUP (GdipImageGetFrameCount);
-  LOOKUP (GdipImageSelectActiveFrame);
-  LOOKUP (GdipGetPropertyItemSize);
-  LOOKUP (GdipGetPropertyItem);
-  LOOKUP (GdipGetPropertyCount);
-  LOOKUP (GdipGetPropertyIdList);
-  LOOKUP (GdipCreateBitmapFromScan0);
-  LOOKUP (GdipSaveImageToStream);
-  LOOKUP (GdipBitmapSetPixel);
-  LOOKUP (GdipDrawImageI);
-  LOOKUP (GdipGetImageGraphicsContext);
-  LOOKUP (GdipFlush);
-  LOOKUP (GdipGraphicsClear);
-  LOOKUP (GdipBitmapSetResolution);
-  LOOKUP (GdipGetImageHorizontalResolution);
-  LOOKUP (GdipGetImageVerticalResolution);
-  LOOKUP (GdipLoadImageFromStream);
-  LOOKUP (GdipDeleteGraphics);
-  LOOKUP (GdipGetImageEncoders);
-  LOOKUP (GdipGetImageEncodersSize);
-  LOOKUP (GdipBitmapLockBits);
-  LOOKUP (GdipBitmapUnlockBits);
-  LOOKUP (GdipGetImagePixelFormat);
-  LOOKUP (GdipCloneBitmapAreaI);
-
-#undef LOOKUP
+  beenhere = TRUE;
 
   input.GdiplusVersion = 1;
   input.DebugEventCallback = NULL;
EOF

patch --verbose --fuzz=0 -p0 <<'EOF' &&
--- gdk-pixbuf/Makefile.in
+++ gdk-pixbuf/Makefile.in
@@ -959,7 +959,7 @@
 	makegdkpixbufalias.pl abicheck.sh pltcheck.sh gdk_pixbuf.def \
 	gdk_pixbuf.rc gdk-pixbuf-marshal.c gdk-pixbuf-marshal.list \
 	gdk-pixbuf-enum-types.c.template \
-	gdk-pixbuf-enum-types.h.template gen-color-table.pl
+	gdk-pixbuf-enum-types.h.template gen-color-table.pl gdiplus.def
 TEST_PROGS = 
 
 ### testing rules
@@ -1137,6 +1137,12 @@
 libpixbufloader_qtif_la_LDFLAGS = -avoid-version -module $(no_undefined)
 libpixbufloader_qtif_la_LIBADD = $(module_libs)
 
+libgdiplus = libgdiplus.dll.a
+gdiplus_ldflag = -Wl,$(libgdiplus)
+
+libgdiplus.dll.a: gdiplus.def
+	$(DLLTOOL) --kill-at --dllname gdiplus.dll --input-def gdiplus.def --output-lib $@
+
 # When building the GDI+ loader statically, we put the "common" objects
 # only in one of the archives to avoid duplicate definitions
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_TRUE@STATIC_GDIPLUS_LIBS = \
@@ -1187,7 +1193,7 @@
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	libpixbufloader-gdip-jpeg.la \
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	libpixbufloader-gdip-tiff.la
 
-@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_ico_la_LDFLAGS = -avoid-version -module -no-undefined
+@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_ico_la_LDFLAGS = -avoid-version -module -no-undefined  $(gdiplus_ldflag)
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_ico_la_SOURCES = \
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-native.h		\
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-propertytags.h		\
@@ -1198,7 +1204,7 @@
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-ico.c
 
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_ico_la_LIBADD = $(module_libs) $(libole32)
-@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_wmf_la_LDFLAGS = -avoid-version -module -no-undefined
+@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_wmf_la_LDFLAGS = -avoid-version -module -no-undefined  $(gdiplus_ldflag)
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_wmf_la_SOURCES = \
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-native.h		\
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-propertytags.h		\
@@ -1209,7 +1215,7 @@
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-wmf.c
 
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_wmf_la_LIBADD = $(module_libs) $(libole32)
-@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_emf_la_LDFLAGS = -avoid-version -module -no-undefined
+@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_emf_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_emf_la_SOURCES = \
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-native.h		\
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-propertytags.h		\
@@ -1220,7 +1226,7 @@
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-emf.c
 
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_emf_la_LIBADD = $(module_libs) $(libole32)
-@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_bmp_la_LDFLAGS = -avoid-version -module -no-undefined
+@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_bmp_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_bmp_la_SOURCES = \
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-native.h		\
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-propertytags.h		\
@@ -1231,7 +1237,7 @@
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-bmp.c
 
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_bmp_la_LIBADD = $(module_libs) $(libole32)
-@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_gif_la_LDFLAGS = -avoid-version -module -no-undefined
+@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_gif_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_gif_la_SOURCES = \
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-native.h		\
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-propertytags.h		\
@@ -1242,7 +1248,7 @@
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-gif.c
 
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_gif_la_LIBADD = $(module_libs) $(libole32)
-@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_jpeg_la_LDFLAGS = -avoid-version -module -no-undefined
+@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_jpeg_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_jpeg_la_SOURCES = \
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-native.h		\
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-propertytags.h		\
@@ -1264,7 +1270,7 @@
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-png.c
 
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_png_la_LIBADD = $(module_libs) $(libole32)
-@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_tiff_la_LDFLAGS = -avoid-version -module -no-undefined
+@BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_tiff_la_LDFLAGS = -avoid-version -module -no-undefined $(gdiplus_ldflag)
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@libpixbufloader_gdip_tiff_la_SOURCES = \
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-native.h		\
 @BUILD_GDIPLUS_LOADERS_TRUE@@INCLUDE_GDIPLUS_FALSE@	io-gdip-propertytags.h		\
@@ -1402,10 +1408,11 @@
 	-version-info $(LT_VERSION_INFO)	\
 	$(LIBTOOL_EXPORT_OPTIONS)		\
 	$(no_undefined)				\
-	$(gdk_pixbuf_symbols)
+	$(gdk_pixbuf_symbols)			\
+	$(gdiplus_ldflag)
 
 libgdk_pixbuf_2_0_la_LIBADD = pixops/libpixops.la $(builtin_objs) $(GDK_PIXBUF_DEP_LIBS) $(libole32)
-libgdk_pixbuf_2_0_la_DEPENDENCIES = pixops/libpixops.la $(builtin_objs) $(gdk_pixbuf_def) $(gdk_pixbuf_win32_res)
+libgdk_pixbuf_2_0_la_DEPENDENCIES = pixops/libpixops.la $(builtin_objs) $(gdk_pixbuf_def) $(gdk_pixbuf_win32_res) $(libgdiplus)
 gdk_pixbuf_headers = \
 	gdk-pixbuf.h			\
 	gdk-pixbuf-core.h		\
EOF

patch --verbose --fuzz=0 -p1 <<'EOF' &&
commit ba5ba3b022d6811c886501fb015e99ce8ee80138
Author: Benjamin Otte <otte@redhat.com>
Date:   Wed Sep 1 00:59:56 2010 +0200

    gdk: Ensure flush of surface when cairo_t is destroyed
    
    Add an ugly workaround because GTK does not ensure surfaces get flushed
    before directly accessing the drawable backed by the surface. This is
    not visible on X11 (where flushing is a no-op), but can be seen on
    Windows.
    
    https://bugzilla.gnome.org/show_bug.cgi?id=628291

diff --git a/gdk/gdkcairo.c b/gdk/gdkcairo.c
index ef8ec25..e1d0daf 100644
--- a/gdk/gdkcairo.c
+++ b/gdk/gdkcairo.c
@@ -23,6 +23,13 @@
 #include "gdkregion-generic.h"
 #include "gdkalias.h"
 
+static void
+gdk_ensure_surface_flush (gpointer surface)
+{
+  cairo_surface_flush (surface);
+  cairo_surface_destroy (surface);
+}
+
 /**
  * gdk_cairo_create:
  * @drawable: a #GdkDrawable
@@ -43,6 +50,7 @@
 cairo_t *
 gdk_cairo_create (GdkDrawable *drawable)
 {
+  static const cairo_user_data_key_t key;
   cairo_surface_t *surface;
   cairo_t *cr;
     
@@ -54,7 +62,12 @@ gdk_cairo_create (GdkDrawable *drawable)
   if (GDK_DRAWABLE_GET_CLASS (drawable)->set_cairo_clip)
     GDK_DRAWABLE_GET_CLASS (drawable)->set_cairo_clip (drawable, cr);
     
-  cairo_surface_destroy (surface);
+  /* Ugly workaround for GTK not ensuring to flush surfaces before
+   * directly accessing the drawable backed by the surface. Not visible
+   * on X11 (where flushing is a no-op). For details, see
+   * https://bugzilla.gnome.org/show_bug.cgi?id=628291
+   */
+  cairo_set_user_data (cr, &key, surface, gdk_ensure_surface_flush);
 
   return cr;
 }
EOF

DEPS=`latest --arch=${ARCH} glib pkg-config atk cairo freetype fontconfig pango libpng zlib`
PROXY_LIBINTL=`latest --arch=${ARCH} proxy-libintl`

PKG_CONFIG_PATH=
for D in $DEPS; do
    PATH=/devel/dist/${ARCH}/$D/bin:$PATH
    [ -d /devel/dist/${ARCH}/$D/lib/pkgconfig ] && PKG_CONFIG_PATH=/devel/dist/${ARCH}/$D/lib/pkgconfig:$PKG_CONFIG_PATH
done

LIBPNG=`latest --arch=${ARCH} libpng`
ZLIB=`latest --arch=${ARCH} zlib`

# Don't do any relinking and don't use any wrappers in libtool. It
# just causes trouble, IMHO.

sed -e 's/need_relink=yes/need_relink=no # no way --tml/' \
    -e 's/wrappers_required=yes/wrappers_required=no # no thanks --tml/' \
    <ltmain.sh >ltmain.temp && mv ltmain.temp ltmain.sh

lt_cv_deplibs_check_method='pass_all' \
CC='gcc -mthreads' \
CPPFLAGS="-I/devel/dist/${ARCH}/${LIBPNG}/include \
-I/devel/dist/${ARCH}/${ZLIB}/include \
-I/devel/dist/${ARCH}/${PROXY_LIBINTL}/include" \
LDFLAGS="-L/devel/dist/${ARCH}/${LIBPNG}/lib \
-L/devel/dist/${ARCH}/${ZLIB}/lib \
-L/devel/dist/${ARCH}/${PROXY_LIBINTL}/lib -Wl,--exclude-libs=libintl.a \
-Wl,--enable-auto-image-base" \
LIBS=-lintl \
LIBPNG=`pkg-config --libs libpng` \
CFLAGS=-O2 \
./configure \
--with-gdktarget=win32 \
--enable-gdiplus \
--with-included-loaders \
--with-included-immodules \
--without-libjasper \
--enable-debug=yes \
--enable-explicit-deps=no \
--disable-gtk-doc \
--disable-static \
--prefix=$TARGET &&

rm gtk/gtk.def &&

PATH="$PWD/gdk-pixbuf/.libs:/devel/target/$HEX/bin:$PATH" make -j3 install &&

PATH="/devel/target/$HEX/bin:$PATH" gdk-pixbuf-query-loaders >/devel/target/$HEX/etc/gtk-2.0/gdk-pixbuf.loaders &&

(echo '# Note: After adding a new separate gdk-pixbuf loader (for instance the svg one)' &&
    echo '# run gdk-pixbuf-query-loaders.exe redirecting its output into this file.' &&
    echo &&
    echo '# Note that the LoaderDir folder below does not name a folder that is' &&
    echo '# expected to exist. It was just a temporary directory used at build time.' &&
    echo &&
    cat $TARGET/etc/gtk-2.0/gdk-pixbuf.loaders ) >$TARGET/etc/gtk-2.0/gdk-pixbuf.loaders.temp &&
    mv $TARGET/etc/gtk-2.0/gdk-pixbuf.loaders.temp $TARGET/etc/gtk-2.0/gdk-pixbuf.loaders &&

grep -v -E 'Automatically generated|Created by' <$TARGET/etc/gtk-2.0/gtk.immodules >$TARGET/etc/gtk-2.0/gtk.immodules.temp &&
    mv $TARGET/etc/gtk-2.0/gtk.immodules.temp $TARGET/etc/gtk-2.0/gtk.immodules &&

./gtk-zip.sh &&

# Package also the gtk-update-icon-cache.exe.manifest
(cd $TARGET && zip /tmp/${MOD}-dev-${VER}.zip bin/gtk-update-icon-cache.exe.manifest) &&

mv /tmp/${MOD}-${VER}.zip /tmp/$RUNZIP &&
mv /tmp/${MOD}-dev-${VER}.zip /tmp/$DEVZIP

) 2>&1 | tee /devel/src/tml/packaging/$THIS.log

(cd /devel && zip /tmp/$DEVZIP src/tml/packaging/$THIS.{sh,log}) &&
manifestify /tmp/$RUNZIP /tmp/$DEVZIP
