﻿using Lucene.Net.QueryParsers.Flexible.Core;
using Lucene.Net.QueryParsers.Flexible.Core.Nodes;
using Lucene.Net.QueryParsers.Flexible.Core.Parser;
using Lucene.Net.QueryParsers.Flexible.Core.Processors;
using Lucene.Net.QueryParsers.Flexible.Standard.Parser;
using Lucene.Net.QueryParsers.Flexible.Standard.Processors;
using Lucene.Net.Search.Spans;
using Lucene.Net.Util;
using NUnit.Framework;
using System;

namespace Lucene.Net.QueryParsers.Flexible.Spans
{
    /*
     * Licensed to the Apache Software Foundation (ASF) under one or more
     * contributor license agreements.  See the NOTICE file distributed with
     * this work for additional information regarding copyright ownership.
     * The ASF licenses this file to You under the Apache License, Version 2.0
     * (the "License"); you may not use this file except in compliance with
     * the License.  You may obtain a copy of the License at
     *
     *     http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     */

    /// <summary>
    /// This test case demonstrates how the new query parser can be used.
    /// <para/>
    /// It tests queries likes "term", "field:term" "term1 term2" "term1 OR term2",
    /// which are all already supported by the current syntax parser (
    /// <see cref="StandardSyntaxParser"/>).
    /// <para/>
    /// The goals is to create a new query parser that supports only the pair
    /// "field:term" or a list of pairs separated or not by an OR operator, and from
    /// this query generate <see cref="SpanQuery"/> objects instead of the regular
    /// <see cref="Search.Query"/> objects. Basically, every pair will be converted to a
    /// <see cref="SpanTermQuery"/> object and if there are more than one pair they will be
    /// grouped by an <see cref="OrQueryNode"/>.
    /// <para/>
    /// Another functionality that will be added is the ability to convert every
    /// field defined in the query to an unique specific field.
    /// <para/>
    /// The query generation is divided in three different steps: parsing (syntax),
    /// processing (semantic) and building.
    /// <para/>
    /// The parsing phase, as already mentioned will be performed by the current
    /// query parser: <see cref="StandardSyntaxParser"/>.
    /// <para/>
    /// The processing phase will be performed by a processor pipeline which is
    /// compound by 2 processors: <see cref="SpansValidatorQueryNodeProcessor"/> and
    /// <see cref="UniqueFieldQueryNodeProcessor"/>.
    /// <para/>
    /// 
    ///     <see cref="SpansValidatorQueryNodeProcessor"/>: as it's going to use the current 
    ///     query parser to parse the syntax, it will support more features than we want,
    ///     this processor basically validates the query node tree generated by the parser
    ///     and just let got through the elements we want, all the other elements as 
    ///     wildcards, range queries, etc...if found, an exception is thrown.
    ///     
    ///     <see cref="UniqueFieldQueryNodeProcessor"/>: this processor will take care of reading
    ///     what is the &quot;unique field&quot; from the configuration and convert every field defined
    ///     in every pair to this &quot;unique field&quot;. For that, a <see cref="SpansQueryConfigHandler"/> is
    ///     used, which has the <see cref="IUniqueFieldAttribute"/> defined in it.
    /// 
    /// <para/>
    /// The building phase is performed by the <see cref="SpansQueryTreeBuilder"/>, which
    /// basically contains a map that defines which builder will be used to generate
    /// <see cref="SpanQuery"/> objects from <see cref="IQueryNode"/> objects.
    /// </summary>
    /// <seealso cref="SpansQueryConfigHandler"/>
    /// <seealso cref="SpansQueryTreeBuilder"/>
    /// <seealso cref="SpansValidatorQueryNodeProcessor"/>
    /// <seealso cref="SpanOrQueryNodeBuilder"/>
    /// <seealso cref="SpanTermQueryNodeBuilder"/>
    /// <seealso cref="StandardSyntaxParser"/>
    /// <seealso cref="UniqueFieldQueryNodeProcessor"/>
    /// <seealso cref="IUniqueFieldAttribute"/>
    public class TestSpanQueryParser : LuceneTestCase
    {
        private QueryNodeProcessorPipeline spanProcessorPipeline;

        private SpansQueryConfigHandler spanQueryConfigHandler;

        private SpansQueryTreeBuilder spansQueryTreeBuilder;

        private ISyntaxParser queryParser = new StandardSyntaxParser();

        public TestSpanQueryParser()
        {
            // empty constructor
        }


        public override void SetUp()
        {
            base.SetUp();

            this.spanProcessorPipeline = new QueryNodeProcessorPipeline();
            this.spanQueryConfigHandler = new SpansQueryConfigHandler();
            this.spansQueryTreeBuilder = new SpansQueryTreeBuilder();

            // set up the processor pipeline
            this.spanProcessorPipeline
                .SetQueryConfigHandler(this.spanQueryConfigHandler);

            this.spanProcessorPipeline.Add(new WildcardQueryNodeProcessor());
            this.spanProcessorPipeline.Add(new SpansValidatorQueryNodeProcessor());
            this.spanProcessorPipeline.Add(new UniqueFieldQueryNodeProcessor());

        }

        public SpanQuery GetSpanQuery(/*CharSequence*/string query)
        {
            return GetSpanQuery("", query);
        }

        public SpanQuery GetSpanQuery(String uniqueField, /*CharSequence*/string query)
        {

            this.spanQueryConfigHandler.Set(SpansQueryConfigHandler.UNIQUE_FIELD, uniqueField);

            IQueryNode queryTree = this.queryParser.Parse(query, "defaultField");
            queryTree = this.spanProcessorPipeline.Process(queryTree);

            return (SpanQuery)this.spansQueryTreeBuilder.Build(queryTree); // LUCENENET TODO: Find way to remove cast

        }

        [Test]
        public void TestTermSpans()
        {
            assertEquals(GetSpanQuery("field:term").toString(), "term");
            assertEquals(GetSpanQuery("term").toString(), "term");

            assertTrue(GetSpanQuery("field:term") is SpanTermQuery);
            assertTrue(GetSpanQuery("term") is SpanTermQuery);

        }

        [Test]
        public void TestUniqueField()
        {
            assertEquals(GetSpanQuery("field", "term").toString(), "field:term");
            assertEquals(GetSpanQuery("field", "field:term").toString(), "field:term");
            assertEquals(GetSpanQuery("field", "anotherField:term").toString(),
            "field:term");

        }

        [Test]
        public void TestOrSpans()
        {
            assertEquals(GetSpanQuery("term1 term2").toString(),
            "spanOr([term1, term2])");
            assertEquals(GetSpanQuery("term1 OR term2").toString(),
            "spanOr([term1, term2])");

            assertTrue(GetSpanQuery("term1 term2") is SpanOrQuery);
            assertTrue(GetSpanQuery("term1 term2") is SpanOrQuery);

        }

        [Test]
        public void TestQueryValidator()
        {

            try
            {
                GetSpanQuery("term*");
                fail("QueryNodeException was expected, wildcard queries should not be supported");

            }
#pragma warning disable 168
            catch (QueryNodeException ex)
#pragma warning restore 168
            {
                // expected exception
            }

            try
            {
                GetSpanQuery("[a TO z]");
                fail("QueryNodeException was expected, range queries should not be supported");

            }
#pragma warning disable 168
            catch (QueryNodeException ex)
#pragma warning restore 168
            {
                // expected exception
            }

            try
            {
                GetSpanQuery("a~0.5");
                fail("QueryNodeException was expected, boost queries should not be supported");

            }
#pragma warning disable 168
            catch (QueryNodeException ex)
#pragma warning restore 168
            {
                // expected exception
            }

            try
            {
                GetSpanQuery("a^0.5");
                fail("QueryNodeException was expected, fuzzy queries should not be supported");

            }
#pragma warning disable 168
            catch (QueryNodeException ex)
#pragma warning restore 168
            {
                // expected exception
            }

            try
            {
                GetSpanQuery("\"a b\"");
                fail("QueryNodeException was expected, quoted queries should not be supported");

            }
#pragma warning disable 168
            catch (QueryNodeException ex)
#pragma warning restore 168
            {
                // expected exception
            }

            try
            {
                GetSpanQuery("(a b)");
                fail("QueryNodeException was expected, parenthesized queries should not be supported");

            }
#pragma warning disable 168
            catch (QueryNodeException ex)
#pragma warning restore 168
            {
                // expected exception
            }

            try
            {
                GetSpanQuery("a AND b");
                fail("QueryNodeException was expected, and queries should not be supported");

            }
#pragma warning disable 168
            catch (QueryNodeException ex)
#pragma warning restore 168
            {
                // expected exception
            }
        }
    }
}
